/*  Copyright (C) MOXA Inc. All rights reserved.

    This software is distributed under the terms of the
    MOXA License.  See the file COPYING-MOXA for details.
*/
/*
    mserial_connection.c

    Routines to operate serial port connections.

    2008-08-15	CF Lin
		new release
*/
#include <ctype.h>
#include "common.h"
#include "mserial_port.h"

static void
mxsp_connection_close(CONNPRT *con)
{
	EVENTMR *enttm = connection_get_metadata(con);
	UARTPRM *param;

	param = (UARTPRM*) connection_get_parameters(con);
	if (param) free(param);

	enttm = connection_get_metadata(con);
	if (enttm)
	{
		if (enttm->event_timer_id) timer_remove(enttm->event_timer_id);
		free(enttm);
	}
	mxsp_close(con->fd);
}

int
mxsp_set_parameters( unsigned int fd, UARTPRM *uart )
{
	int n;

	n = mxsp_set_baudrate(fd, uart->baudrate);
	n = mxsp_set_parity(fd, uart->parity);
	n = mxsp_set_databits(fd, uart->data_bits);
	n = mxsp_set_stopbits(fd, uart->stop_bits);
	n = mxsp_set_interface(fd, uart->iface_mode);
	n = mxsp_set_flow_control(fd, uart->flow_control);
	return n;
}

void
mxsp_connection_purge(MHANDLE *hndl)
{
    CONNPRT *conp = (CONNPRT*) hndl;

    mxsp_purge_buffer(conp->fd, MSP_PURGE_RXTX);
}

static void
mxsp_connection_event_timer(void *arg)
{
	CONNPRT *conp = (CONNPRT*) arg;
	EVENTMR *uart;
	unsigned int error = 0, status = 0, event = 0;

	uart = connection_get_metadata(conp);
	if (uart)
	{
		mxsp_get_errors(conp->fd,  &error);
		mxsp_get_lstatus(conp->fd, &status);
		event = ((error<<16)+status) & uart->event_mask;
		if (uart->event != event)	
		{
			uart->event_callback(conp, event);
			uart->event = event;
		}
	}
}

int
mxsp_connection_set_mask(MHANDLE *hndl, conn_event_t cb, unsigned int mask, unsigned int interval)
{
    CONNPRT *conp = (CONNPRT*) hndl;
	EVENTMR *uart;

	if (cb==NULL || interval==0)
		return -1;

	uart = connection_get_metadata(conp);
	if (uart)
	{
		if (uart->event_timer_id == 0)
		{
			uart->event_callback = cb;
			uart->event_timer_id = connection_add_timer(mxsp_connection_event_timer, interval, conp);
		}
		else
			timer_set_interval(uart->event_timer_id, interval);
		uart->event_mask = mask;
	}
	return 0; //mxsp_set_mask(conp->fd, mask);
}

/*  make a serial port connection and add this connection 
    into the golbal connection pool
	Inputs:
		<uart> the paramters of the port
		<funs> user-defined functions
	Returns:
		the connection
*/
MHANDLE
mxsp_connection_open (UARTPRM *param, USERFUN *funs)
{
	unsigned int fd;
	CONNPRT *xp = NULL;

    if (!param)
		return NULL;

dbgprintf("++mxsp_connection_open: port (%d)", param->port);

	/* open the COM port */
	fd = mxsp_open(param->port);
	if (fd == 0)
	{
		dbgprintf("make_mxsp_client: fail to open COM port %d", param->port);
	}
	else
	{
		unsigned int options = 0;

		param = (UARTPRM*) my_malloc(param,sizeof(UARTPRM));
		if (!param)
		{
			mxsp_close(fd);
			return NULL;
		}
		if (COM_PORT_SELECTABLE)
			options = OPTION_SELECTABLE|OPTION_CALL_OPEN;
		else
			options = OPTION_CALL_OPEN;

		/* change the communication parameters */
		mxsp_set_parameters(fd, param);
		/* add this client into the pool */
		xp = connection_add(CONNECTION_TYPE_UARTPORT, fd, param->port, options, funs, param, NULL,
			mxsp_read, mxsp_write, mxsp_connection_close);
		if (xp)
		{
dbgprintf("--mxsp_connection_open: port (%d)", param->port);
		}
	}
	return xp;
}

/***** added by Carl on 2009-08-05 *****/
int
mxsp_get_parameters( unsigned int fd, UARTPRM *uart )
{
	uart->baudrate = mxsp_get_baudrate(fd);
	uart->parity = mxsp_get_parity(fd);
	uart->data_bits = mxsp_get_databits(fd);
	uart->stop_bits = mxsp_get_stopbits(fd);
	uart->iface_mode = mxsp_get_interface(fd);
	uart->flow_control = mxsp_get_flow_control(fd);

	return 0;
}

int
mxsp_connection_set_parameters( CONNPRT *con, UARTPRM *uart )
{
	return mxsp_set_parameters( con->fd, uart );
}

int
mxsp_connection_get_parameters( CONNPRT *con, UARTPRM *uart )
{
	return mxsp_get_parameters( con->fd, uart );
}

#if 0
MHANDLE
mxsp_connection_open_default (UARTPRM *param, USERFUN *funs)
{
	unsigned int fd;
	CONNPRT *xp = NULL;

    if (!param)
		return NULL;

dbgprintf("++mxsp_connection_open_default: port (%d)", param->port);

	/* open the COM port */
	fd = mxsp_open_default(param->port);
	if (fd == 0)
	{
		dbgprintf("make_mxsp_client: fail to open COM port %d", param->port);
	}
	else
	{
		unsigned int options = 0;

		param = (UARTPRM*) my_malloc(param,sizeof(UARTPRM));
		if (!param)
		{
			mxsp_close(fd);
			return NULL;
		}
		if (COM_PORT_SELECTABLE)
			options = OPTION_SELECTABLE|OPTION_CALL_OPEN;
		else
			options = OPTION_CALL_OPEN;

		/* change the communication parameters */
		//mxsp_set_parameters(fd, param);

		/* add this client into the pool */
		xp = connection_add(CONNECTION_TYPE_UARTPORT, fd, param->port, options, funs, param, NULL,
			mxsp_read, mxsp_write, mxsp_connection_close);
		if (xp)
		{
dbgprintf("--mxsp_connection_open_default: port (%d)", param->port);
		}
	}
	return xp;
}
#endif
/**********************************************/
